<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\User;
use App\Models\Department;
use App\Models\TeamMember;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;

class DepartmentControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $admin;
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        // Create admin user
        $this->admin = User::factory()->create(['role' => 'admin']);

        // Create regular user
        $this->user = User::factory()->create(['role' => 'user']);
    }

    /** @test */
    public function admin_can_create_department()
    {
        $this->actingAs($this->admin);

        $departmentData = [
            'name' => 'Test Department',
            'color_code' => '#FF0000'
        ];

        $response = $this->postJson('/api/departments', $departmentData);

        $response->assertStatus(201)
            ->assertJsonFragment(['name' => 'Test Department']);

        $this->assertDatabaseHas('departments', $departmentData);
    }

    /** @test */
    public function department_can_have_parent_department()
    {
        $this->actingAs($this->admin);

        $parent = Department::factory()->create();

        $childData = [
            'name' => 'Child Department',
            'parent_id' => $parent->id,
            'color_code' => '#00FF00'
        ];

        $response = $this->postJson('/api/departments', $childData);

        $response->assertStatus(201);
        $this->assertDatabaseHas('departments', $childData);
    }

    /** @test */
    public function non_admin_cannot_create_department()
    {
        $this->actingAs($this->user);

        $departmentData = [
            'name' => 'Test Department',
            'color_code' => '#FF0000'
        ];

        $response = $this->postJson('/api/departments', $departmentData);

        $response->assertStatus(403);
    }

    /** @test */
    public function department_name_must_be_unique()
    {
        $this->actingAs($this->admin);

        $department = Department::factory()->create();

        $response = $this->postJson('/api/departments', [
            'name' => $department->name,
            'color_code' => '#FF0000'
        ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['name']);
    }

    /** @test */
    public function can_list_departments_with_hierarchy()
    {
        $this->actingAs($this->user);

        $parent = Department::factory()->create();
        $child = Department::factory()->create(['parent_id' => $parent->id]);

        $response = $this->getJson('/api/departments');

        $response->assertStatus(200)
            ->assertJsonCount(1, 'data') // Only parent departments in root level
            ->assertJsonPath('data.0.children.0.id', $child->id);
    }

    /** @test */
    public function cannot_delete_department_with_children()
    {
        $this->actingAs($this->admin);

        $parent = Department::factory()->create();
        Department::factory()->create(['parent_id' => $parent->id]);

        $response = $this->deleteJson("/api/departments/{$parent->id}");

        $response->assertStatus(422);
        $this->assertDatabaseHas('departments', ['id' => $parent->id]);
    }

    /** @test */
    public function can_delete_department_without_children()
    {
        $this->actingAs($this->admin);

        $department = Department::factory()->create();

        $response = $this->deleteJson("/api/departments/{$department->id}");

        $response->assertStatus(200);
        $this->assertDatabaseMissing('departments', ['id' => $department->id]);
    }
}
