<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\User;
use App\Models\Department;
use App\Models\TeamMember;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;

class TeamMemberControllerTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $admin;
    protected $user;
    protected $department;

    protected function setUp(): void
    {
        parent::setUp();

        $this->admin = User::factory()->create(['role' => 'admin']);
        $this->user = User::factory()->create(['role' => 'user']);
        $this->department = Department::factory()->create();
    }

    /** @test */
    public function admin_can_create_team_member()
    {
        $this->actingAs($this->admin);

        $memberData = [
            'name' => 'John Doe',
            'role' => 'member',
            'department_id' => $this->department->id
        ];

        $response = $this->postJson('/api/team-members', $memberData);

        $response->assertStatus(201)
            ->assertJsonFragment(['name' => 'John Doe']);

        $this->assertDatabaseHas('team_members', $memberData);
    }

    /** @test */
    public function non_admin_cannot_create_team_member()
    {
        $this->actingAs($this->user);

        $memberData = [
            'name' => 'John Doe',
            'role' => 'member',
            'department_id' => $this->department->id
        ];

        $response = $this->postJson('/api/team-members', $memberData);

        $response->assertStatus(403);
    }

    /** @test */
    public function team_member_requires_valid_department()
    {
        $this->actingAs($this->admin);

        $memberData = [
            'name' => 'John Doe',
            'role' => 'member',
            'department_id' => 999 // Non-existent department
        ];

        $response = $this->postJson('/api/team-members', $memberData);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['department_id']);
    }

    /** @test */
    public function team_member_requires_valid_role()
    {
        $this->actingAs($this->admin);

        $memberData = [
            'name' => 'John Doe',
            'role' => 'invalid_role',
            'department_id' => $this->department->id
        ];

        $response = $this->postJson('/api/team-members', $memberData);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['role']);
    }

    /** @test */
    public function can_list_team_members_with_department()
    {
        $this->actingAs($this->user);

        $member = TeamMember::factory()->create([
            'department_id' => $this->department->id
        ]);

        $response = $this->getJson('/api/team-members');

        $response->assertStatus(200)
            ->assertJsonFragment([
                'name' => $member->name,
                'department_id' => $this->department->id
            ]);
    }

    /** @test */
    public function can_filter_team_members_by_department()
    {
        $this->actingAs($this->user);

        $department2 = Department::factory()->create();

        $member1 = TeamMember::factory()->create([
            'department_id' => $this->department->id
        ]);

        $member2 = TeamMember::factory()->create([
            'department_id' => $department2->id
        ]);

        $response = $this->getJson("/api/team-members?department_id={$this->department->id}");

        $response->assertStatus(200)
            ->assertJsonFragment(['name' => $member1->name])
            ->assertJsonMissing(['name' => $member2->name]);
    }

    /** @test */
    public function can_search_team_members_by_name()
    {
        $this->actingAs($this->user);

        $member1 = TeamMember::factory()->create([
            'name' => 'John Doe',
            'department_id' => $this->department->id
        ]);

        $member2 = TeamMember::factory()->create([
            'name' => 'Jane Smith',
            'department_id' => $this->department->id
        ]);

        $response = $this->getJson('/api/team-members?search=John');

        $response->assertStatus(200)
            ->assertJsonFragment(['name' => 'John Doe'])
            ->assertJsonMissing(['name' => 'Jane Smith']);
    }

    /** @test */
    public function admin_can_update_team_member()
    {
        $this->actingAs($this->admin);

        $member = TeamMember::factory()->create();
        $newDepartment = Department::factory()->create();

        $updateData = [
            'name' => 'Updated Name',
            'role' => 'manager',
            'department_id' => $newDepartment->id
        ];

        $response = $this->putJson("/api/team-members/{$member->id}", $updateData);

        $response->assertStatus(200)
            ->assertJsonFragment($updateData);

        $this->assertDatabaseHas('team_members', $updateData);
    }

    /** @test */
    public function admin_can_delete_team_member()
    {
        $this->actingAs($this->admin);

        $member = TeamMember::factory()->create();

        $response = $this->deleteJson("/api/team-members/{$member->id}");

        $response->assertStatus(200);
        $this->assertDatabaseMissing('team_members', ['id' => $member->id]);
    }
}
