# Performance Notes

This document outlines performance considerations and potential optimizations for the mailing system.

## 1. IMAP Fetch Strategies

- **Current Strategy**: The system currently fetches all messages from the INBOX and then limits the result to 50. This is inefficient as it requires the server to process all messages before the limit is applied. The `webklex/php-imap` library fetches message headers first, and then bodies are loaded somewhat lazily, but it can still be slow on very large mailboxes.
- **Chunking/Pagination**: The current implementation lacks proper pagination. For large mailboxes, fetching an unlimited number of messages initially can lead to high memory usage and long response times. A proper pagination strategy (e.g., `->paginate(per_page: 25, page: 1)`) should be implemented.
- **Lazy Body Loading**: The `fetch_body` option in `config/imap.php` is `true`. For the inbox view, which only shows snippets, it would be more performant to set this to `false` and only fetch the full body when the user clicks to view the detail of a specific email.

## 2. Timeouts and Retries

- **Timeouts**: The `timeout` option in `config/imap.php` is set to 30 seconds. This is a reasonable default, but for slow connections or very large messages, it could lead to exceptions. 
- **Retries**: There is no automatic retry mechanism. A failed IMAP connection or fetch operation will result in an immediate error shown to the user. For background jobs, a retry strategy with exponential backoff would be essential.

## 3. Caching

- **Current State**: There is **no caching**. Every request to the inbox results in a live call to the IMAP server.
- **Impact**: This is the single biggest performance bottleneck. It leads to slow page loads, high bandwidth usage, and unnecessary load on the IMAP server.
- **Recommendations**:
    - **Implement a Database Cache**: Use a database schema like the one proposed in `data-model.md` to store email headers and bodies.
    - **Cache Index/Snippets**: For the inbox view, only the essential fields (from, subject, date, snippet, flags) should be read from the cache. This would make inbox loading nearly instantaneous.
    - **DB Indexing**: If a caching schema is implemented, columns that are frequently searched or sorted (e.g., `user_id`, `folder`, `date`, `uid`) must be indexed in the database.

## 4. Attachment Handling

- **Current State**: Attachment handling is not implemented.
- **Impact**: When implemented, fetching large attachments directly from the IMAP server on-the-fly will be very slow and memory-intensive.
- **Recommendations**:
    - **Size Limits**: Impose a reasonable size limit on attachments that can be fetched directly.
    - **Streaming**: For large attachments, stream the download directly from the IMAP server to the user's browser without loading the entire file into the application's memory.
    - **Cache Attachments**: If caching is implemented, attachments should be stored on a filesystem (local or cloud storage like S3) and served from there, which is much more efficient than repeated IMAP fetches.

## 5. Queues & Background Jobs

- **Current State**: There are no queues or jobs for email processing. All actions happen synchronously during the user's web request.
- **Impact**: This leads to a poor user experience, as actions like syncing a mailbox or sending an email (on a slow SMTP server) will block the UI.
- **Recommendations**:
    - **Background Sync**: Create a job that runs periodically (e.g., every 15 minutes) to sync mailboxes and update the cache. This is far more efficient than live fetching.
    - **Use `imap:idle`**: The `webklex/laravel-imap` package includes a command `imap:idle` which can be run as a long-lived process to get near real-time email notifications. This is more advanced but very powerful.
    - **Queue Outgoing Mail**: For sending emails, dispatch a job to a queue. This will make the UI feel instant to the user, and the email will be sent by a background worker.
